/**
 * Parallel computing of pi in using rectangle rule - RMA One side communication
 * The total rectangles number is 
 *   - read from keyboard when the program is executed on command line
 *   - read from a text file "rectsNb.txt" when SLURM is used
 * 
 * The task of rank 0 
 *   - reads the total rectangles number (n)
 *   - creates the shared memory windows for n and pi
 *   - computes a partial result
 *   - print the result when it is complete
 *  
 * The task of rank != 0
 *   - get n from rank 0's n memory window
 *   - computes a partial result
 *   - add its result to rank 0's pi memory window
 * 
 *  * Compiling: mpicc pi_rectangle_RMA.c -o pi_rectangle -lm
 */
#include <stdio.h>
#include <math.h>
#include <mpi.h>

int  main(int argc, char **argv)
{
    double PI25DGT = 3.141592653589793238462643;
    long   n=200, rects_per_proc=0, i;
    double x, h, my_sum=0.0, pi=0.0;

    long   my_deb=0, my_end=0;
    int    myRank, nbProcs;
    
    MPI_Win win_pi, win_n;

    MPI_Init( &argc, &argv );
    MPI_Comm_rank( MPI_COMM_WORLD, &myRank );
    MPI_Comm_size( MPI_COMM_WORLD, &nbProcs);

    if (myRank==0) {  // Process of rank 0 does I/O 
        printf("Please input the number of rectangles for [0, 1]: ");
        fflush(stdout);
        scanf("%ld%*c", &n);
    }
   
    if (myRank==0) { // Create the shared memory window
        MPI_Win_create(&n, sizeof(long), sizeof(long),
            MPI_INFO_NULL, MPI_COMM_WORLD, &win_n);
        MPI_Win_create(&pi, sizeof(double), sizeof(double),
            MPI_INFO_NULL, MPI_COMM_WORLD, &win_pi);
    }
    else {	
        MPI_Win_create(MPI_BOTTOM, 0, sizeof(long),
            MPI_INFO_NULL, MPI_COMM_WORLD, &win_n);
        MPI_Win_create(MPI_BOTTOM, 0, sizeof(double),
            MPI_INFO_NULL, MPI_COMM_WORLD, &win_pi);
    }

    MPI_Win_fence(0, win_n);
    if (myRank != 0) 
        MPI_Get(&n, 1, MPI_LONG, 0, 0, 1, MPI_LONG, win_n);
    MPI_Win_fence(0, win_n);
    
   // printf("Proc %d : n=%ld\n", myRank, n);

    rects_per_proc = n / nbProcs;
    h = 1.0 / (rects_per_proc*nbProcs);
 
    my_deb=myRank*rects_per_proc;  
    my_end=my_deb+rects_per_proc;
 
    for (i=my_deb; i<my_end; i++) {
        x = (i+0.5)*h;         
        my_sum += 4.0 / (1.0 + x*x);
    }
    pi = my_sum * h; 
 
    MPI_Win_fence(0, win_pi);
    if (myRank)
        MPI_Accumulate(&pi, 1, MPI_DOUBLE, 0, 0, 1, 
                        MPI_DOUBLE, MPI_SUM, win_pi);
    MPI_Win_fence(0, win_pi);

    if (myRank==0) { // Process of rank 0 print the result on the screen
        printf("pi is approximatly: %0.16f\n", pi);
        printf("Error to PI25DGT is %.16f\n", fabs(pi-PI25DGT));
    }

    MPI_Win_free(&win_n);   // collective communication
    MPI_Win_free(&win_pi);

    MPI_Finalize();
    return 0;
}

